;;; Ystok-Media - Media player by means of ActiveX (OLE pane) on LispWorks for Windows
;;; Copyright (c) 2025 Dr. Dmitry Ivanov. All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Windows Media Player support
;;; https://learn.microsoft.com/en-us/previous-versions/windows/desktop/wmp/windows-media-player-sdk

(in-package :ystok.media)

(defparameter *wmp-play-states*
 #(:undefined	;0  Player is in an undefined state.
   :stop	;1 "Stopped"	Playback of the current media item is stopped.
		;	Called either after (play-state-change 8 and 9) when forward
                ;	or just after (state-change 5) before (state-change 8 and 9)
   :pause	;2 "Paused"	Playback of the current media item is paused.
		;	   	Resuming begins from the same position.
   :play    	;3 "Playing"	The current item is playing.
   :forward	;4 "ScanForward" The current item is fast forwarding.
   :backward	;5 "ScanReverse" The current item is fast rewinding.
   :buffer	;6 "Buffering"	The current item is getting additional data from the server
   :waiting  	;7 "Waiting"	Connection is established, but the server is not
      		;		sending data. Waiting for session to begin.
   :end		;8 "MediaEnded"	Media item has completed playback.
		; 	   BAD:	On ending a video, sometimes keeps displaying.
                ; 		On ending a photo, the sceen turns black.
		;          NB:	Acually this is very shot event as (state-change 9) (... 1)
                ;		are immediately follows automatically.
   :trans	;9 "Transitioning" Preparing new media item.
   :ready	;10 "Ready"	Ready to begin playing.
   :connect	;11 "Reconnecting" Reconnecting to stream.
) )


(define-vendor :wmp :title "Windows Media Player"	; version 7 or higher
               :component-name "WMPlayer.OCX"		;"WMPlayer.OCX.7"
               :player-class 'wm-player)
               ;:sink-callback 'wmp-callback)

(defclass wm-player (player)
 ((vendor-id :allocation :class :initform :wmp)
  (%ui-mode% :initform nil)			; desireable mode :none :mini or :full
  ;; Slots holding innstances of com:com-interface i-dispatch for invoking
  ;; the OLE properties and methods for inner objects
  (i-controls :initform nil)			; IWMPControls
  (i-settings :initform nil)			; IWMPSettings
  ;; IWMPMedia or NIL (means :nothing); should be dropped during next/previous
  ;; May not be eq to any of the i-media slot values among (itmes player)
  ;; but usually vendor-i-media-eq to some of them.
  (i-current-media :initform nil)))
  ;(play-states :initform (cons 0 0))		; (last . butlast)

;;; I-DISPATCH getters

(defmethod i-playlist (pane (player wm-player))
 ;;; Value: IWMPPlaylist
  (com:invoke-dispatch-get-property (i-dispatch pane player) "currentPlaylist"))

(defun i-controls (pane &optional (player (player pane)))
  (or (and player (slot-value player 'i-controls))
      (let ((i (com:invoke-dispatch-get-property (i-dispatch pane player) "controls")))
        (if player (setf (slot-value player 'i-controls) i) i))))

(defun i-settings (pane &optional (player (player pane)))
  (or (and player (slot-value player 'i-settings))
      (let ((i (com:invoke-dispatch-get-property (i-dispatch pane player) "settings")))
        (if player (setf (slot-value player 'i-settings) i) i))))

(defun i-current-media (pane &optional (player (player pane)))
 ;;; Value: IWMPMedia object
  ;;        NIL if there is no media opened or the playlist is empty.
  ;; WMP BAD: currentMedia returns some media object even if the current playlist is empty!
  (or (slot-value player 'i-current-media)
      (when (< 0 (vendor-items-count player pane))
        (when-let (i (com:invoke-dispatch-get-property (i-controls pane player)
                                                       "currentItem"))
					;(... (i-dispatch pane player) "currentMedia")
          (unless (eq i :nothing)					; no media opened
            (setf (slot-value player 'i-current-media) i))))))

(defmethod vendor-i-eq ((player wm-player) (x com:com-interface) (y com:com-interface))
 ;;; Args: x,y  COM-interfaces of media or playlist
  (com:invoke-dispatch-method y "isIdentical" x))

(defmethod vendor-component-on-insert ((player wm-player) pane)
 ;;; UI mode is "full" (default) is buggy.
  ;; 1) If it is set before opening a file the very first time,
  ;;	the following does not work for me:
  ;;	- video is not rendered,
  ;;	- controls are shown but not available.
  ;;	After opening a video file, you can change uiMode and everything works.
  ;; 2) When paused, moving the photor within the seekbar has no effect on picture.
  (let* ((plist (CAPI:CAPI-OBJECT-PLIST pane))
         (ui-mode (getf plist :ui-mode *ui-mode*)))
    (setf (vendor-ui-mode player pane :insert t) (if (eq ui-mode :full) :mini ui-mode)
          (slot-value player '%ui-mode%) ui-mode)			; memoize as is
          ;(slot-value player 'i-playlist)				; temp playlist
          ; (com:invoke-dispatch-method (i-dispatch pane player) "newPlaylist" "" nil))
    ;; Displayed video automatically sizes to fit the video window
    ;; but looks rather reduced when ui-mode :mini until the very frist media is added.
    (com:invoke-dispatch-put-property (i-dispatch pane player) "stretchToFit" t)))
    ;(com:invoke-dispatch-put-property (i-dispatch pane player) "windowlessVideo" 1)

(defmethod vendor-component-on-close ((player wm-player) pane)
  (setf (slot-value player '%ui-mode%) nil)
  (dolist (slot-name '(i-controls i-current-media i-settings))
    (when-let (i (slot-value player slot-name))
      (com:release i)
      (setf (slot-value player slot-name) nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PLAYLIST, ITEMS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;; BAD: "next" and "previous" seem not working

(defmethod vendor-add ((player wm-player) pane
                       &key pathname title
                            (clear (vendor-get player pane :singleton *singleton*))
                            (photo-duration *photo-duration*))
 ;;; To load a Media item using a file name, set the URL property or use newMedia
  ;; Args: photo-duration
  ;;		 Time in seconds passed as :image-duration option a la
  ;;		 Settings -> Input/codecs / Demusers -> Image
  ;;		 ( -> / ->  -> )
  ;;       other-keys
  ;;       	 Must be passed to COM-method in a string separated by space, e.g.
  ;;       	 ":aspect-ratio=4:3 --rtsp-tcp"
  ;; NB: For singleton playlist, the following works fine
  ;;	(com:invoke-dispatch-put-property i-dispatch "URL" url)
  (let* ((url (namestring pathname))			; string-append "file:///"
         (i-dispatch (i-dispatch pane player))
         (i-playlist (i-playlist pane player))
	 ;; Hard-wire photo duration as vendor-duration always returns 0 for it.
         (duration (if (photo-p pathname) photo-duration nil)))
    ;; Workaround bug (in version 12 at least): full mode does not work on insert-component
    (when (and (eq (slot-value player '%ui-mode%) :full)
               (not (eq (vendor-ui-mode player pane) :full)))
      (setf (vendor-ui-mode player pane) :full))
    (when clear
      (vendor-clear player pane))
    (when-let (i-media (com:invoke-dispatch-method i-dispatch "newMedia" url))
      ;(when duration						; no working even here
      ; (com:invoke-dispatch-put-property i-media "duration" (float duration 1d0))
      ; (com:invoke-dispatch-method i-media "setItemInfo" "Duration" (float duration 1d0))
      (let ((item (make-instance 'item
                                 :title (or title (pathname-name pathname))
                                 :url pathname
                                 :duration duration
                                 :media i-media)))
        (push-end item (items player))
        (when (< (index player) 0)				; simplest synchronization
          (setf (index player) 0))
        (com:invoke-dispatch-method i-playlist "appendItem" i-media)
        #+debug (assert (eql (length (items player)) (vendor-items-count player pane) ))
        item))))

(defmethod vendor-remove ((player wm-player) pane &key index (item (item player index)))
 ;;; If the item removed is the currently playing track (Player.currentMedia),
  ;; playback stops and the next item in the playlist becomes the current one.
  ;; If there is no next item, the previous item is used, or if there are no other items,
  ;; then Player.currentMedia is set to NULL.
  ;; NB: (wmp-current-playlist-change 4)  usually occurs after wmp-current-item-change
  ;; CAUTION: Passing (i-media item) raises exception!
  ;; BAD: Method "item" signals in a delivered application.
  (cond ((null item)
         nil)
        ((if index 
             (< -1 index (length (items player)))
             (setq index (position item (items player))))
         (let* ((pos (index player))				; current index
                (i (ignore-errors (com:invoke-dispatch-method	; signals in delivered!
                                   (i-playlist pane player) "item" index))))
          (if i
              (progn
                (setf (slot-value player 'i-current-media) nil)
                (cond ((null (removef (items player) item))	; no item will stay in
                       (setf (index player) -1			; - a la vendor-clear
                             (state player) nil
                             (status player) nil))
                      ((< index pos)				; removed preceeds current
                       (decf (index player)))			; decrease current index
                      ((= pos index (length (items player)))	; removing current at end
                       (setf (index player) 0)))		; first is current now
                (com:invoke-dispatch-method (i-playlist pane player)
                                            "removeItem" i)) ;-> wmp-current-item-change
              (capi:beep-pane pane))))
        (t (capi:beep-pane pane))))

(defmethod vendor-clear ((player wm-player) pane)
  ;(when-let (i (slot-value player 'i-current-media))
  ;  (com:release i)
  (setf (slot-value player 'i-current-media) nil)
        ;(slot-value player 'play-states) (cons 0 0))
  ;(com:invoke-dispatch-put-property (i-dispatch pane player) "currentMedia" nil) ; no work
  (com:invoke-dispatch-method (i-playlist pane player) "clear"))
 #|(for with i-playlist = (i-playlist pane player)
    for item in items (items player) 
    and index of-type fixnum upfrom 0 below (vendor-items-count player pane)
    for i-media = (com:invoke-dispatch-method i-playlist "item" index)
    for i-media = (i-media item)
    when i-media
      do (com:invoke-dispatch-method i-playlist "removeItem" i-media)))|#

(defmethod vendor-items-count ((player wm-player) pane)
  (com:invoke-dispatch-get-property (i-playlist pane player) "count"))

(defmethod vendor-next ((player wm-player) pane)
  (let ((i-controls (i-controls pane player)))
    (if (com:invoke-dispatch-get-property i-controls "isAvailable" #1="next")
        (com:invoke-dispatch-method (i-controls pane player) #1#)
        (capi:beep-pane pane))))

(defmethod vendor-previous ((player wm-player) pane)
  (let ((i-controls (i-controls pane player)))
    (if (com:invoke-dispatch-get-property i-controls "isAvailable" #1="previous")
        (com:invoke-dispatch-method (i-controls pane player) #1#)
        (capi:beep-pane pane))))

(defmethod vendor-set-current-item (player pane &key index (item (item player index)))
 ;;; Change playlit item without forcing to play it
  ;; Value: item on success.
  ;; NB: Fires event -> "CurrentItemChange"
  ;; UNUSED: vendor-play does the same.
  (cond ((null item)
         nil)
        ((if index 
             (< -1 index (length (items player)))
             (setq index (position item (items player))))
         (setf (index player) index					; for safety
               (slot-value player 'i-current-media) nil)		; drop cache?
         #+ys-product (yl:debug-format :media "Current item := ~s pos ~s" item index)
         (com:invoke-dispatch-put-property (i-controls pane player) "currentItem"
           (com:invoke-dispatch-method (i-playlist pane player) "item" index))
                                            ;(i-media item))
          ;; BAD: Assinging "currentMedia" seems to implicitly create a new playlist.
          ;(com:invoke-dispatch-put-property(i-dispatch pane player)"currentMedia" i-media)
         item)
        (t (capi:beep-pane pane)) ))					; something wrong
 
(defmethod vendor-item-descripton ((player wm-player) pane &key (item (item player)))
 ;;; getItemInfo returns a String representing the value of the specified attribute.
  ;; For Boolean, it returns the string "true" or "false".
  (when-let (i-media (or (slot-value player 'i-current-media)
                         (and item (i-media item))))
                          ;(and (or (null item) (current-p item)))?
    (format nil "Title ~a~%Artist ~a~%FileSize ~s~%Type ~s"
            (com:invoke-dispatch-method i-media "getItemInfo" "Title")
            (com:invoke-dispatch-method i-media "getItemInfo" "Author") ; "artist"
            (com:invoke-dispatch-method i-media "getItemInfo" "FileSize")
            ;; "audio", "other", "photo", "playlist", "radio", or "video"
	    (com:invoke-dispatch-method i-media "getItemInfo" "MediaType"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PROPERTIES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
(defmethod vendor-get ((player wm-player) pane  (prop (eql :autoloop)) &optional default)
  (com:invoke-dispatch-method (i-settings pane player) "getMode" "loop"))

(defmethod vendor-put ((player wm-player) pane (prop (eql :autoloop)) value)
  (com:invoke-dispatch-method (i-settings pane player) "setMode" "loop" value))

(defmethod vendor-get ((player wm-player) pane (prop (eql :autostart)) &optional default)
  (com:invoke-dispatch-get-property (i-settings pane player) "autoStart"))

(defmethod vendor-put ((player wm-player) pane (prop (eql :autostart)) value)
  (com:invoke-dispatch-put-property (i-settings pane player) "autoStart" value))

(defmethod vendor-get ((player wm-player) pane (prop (eql :rate)) &optional default)
  (com:invoke-dispatch-get-property (i-settings pane player) "rate"))

(defmethod vendor-put ((player wm-player) pane (prop (eql :rate)) (value number))
  (let ((i-settings (i-settings pane player)))
    (if (com:invoke-dispatch-get-property i-settings "isAvailable" "Rate")
        (com:invoke-dispatch-put-property i-settings "rate" (float value))
        (capi:beep-pane pane))))

(defmethod vendor-ui-mode ((player wm-player) pane)
  (when-let (pair (assoc (com:invoke-dispatch-get-property (i-dispatch pane player)
                                                           "uiMode")
                         *ui-mode-alist*
                         :test #'string-equal))
    (car pair)))
    
(defmethod (setf vendor-ui-mode) (value (player wm-player) pane &key insert)
  (when (and (keywordp value) (assoc value *ui-mode-alist* :test #'eq))
    (com:invoke-dispatch-put-property (i-dispatch pane player) "uiMode"
                                      (string-downcase value))
    (unless insert
      (setf (slot-value player '%ui-mode%) value))			; syncronize
    value))

(defmethod vendor-duration ((player wm-player)
                            &optional pane (i-media (i-current-media pane player)))
  (when i-media
    (when-let (sec (com:invoke-dispatch-get-property i-media "duration"))
      (if (< 0 sec) (round sec) nil))))

(defmethod vendor-duration-string ((player wm-player)
                                   &optional pane (i-media (i-current-media pane player)))
  (if i-media
      (com:invoke-dispatch-get-property i-media "durationString")
      ""))

(defmethod vendor-time ((player wm-player) &optional pane)
 ;;; Property "currentPosition" provides time item in seconds from beginning.
  (let ((sec (com:invoke-dispatch-get-property (i-controls pane player)"currentPosition")))
    (if (and sec (<= 0 sec)) (round sec) nil)))

(defmethod (setf vendor-time) ((sec number) (player wm-player) &optional pane)
  (com:invoke-dispatch-put-property (i-controls pane player) "currentPosition"
                                    (float sec 1d0)))

(defmethod vendor-timestring ((player wm-player) &optional pane)
 ;;; If the media item is less than an hour long, the HH: portion is not included.
  (com:invoke-dispatch-get-property (i-controls pane player)"currentPositionString"))

(defmethod (setf vendor-position) ((value float) (player wm-player) &optional pane)
 ;;; STUB
  (if (zerop value)
      (vendor-rewind player pane)
      (call-next-method)))

(defmethod vendor-state ((player wm-player)
                         &optional pane (index (com:invoke-dispatch-get-property
                                                (i-dispatch pane) "playState")))
  (when (< -1 index (load-time-value (length *wmp-play-states*)))
    (svref *wmp-play-states* index)))

(defmethod vendor-status ((player wm-player) &optional pane)
  (com:invoke-dispatch-get-property (i-dispatch pane) "status"))

(defmethod vendor-version ((player wm-player) &optional pane)	; "getVersionInfo" signals
  (let ((i (i-dispatch pane player)))
    (string-append (com:invoke-dispatch-get-property i "versionInfo")
                   #\Newline
                   (com:invoke-dispatch-get-property i "status")))) ; localized

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  METHODS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod vendor-play ((player wm-player) pane &key index (item (item player index)))
 ;;; Args: index, item
  ;;	      -	Load and play automatically, regardless of the value of Settings.autoStart
  ;;	      -	Fire event -> wmp-current-item-change even if the currentItem is the same.
  ;; BAD: Simply passing (i-media item) to "playItem" raises
  ;;	  COM IDispatch::Invoke Exception  in "IWMPControls3::playItem" code -1072885626
  (let ((i-controls (i-controls pane player)))
    (cond (item
           (if (if index 
                   (< -1 index (length (items player)))
                   (setq index (position item (items player))))
               (progn (setf (index player) index)			; for safety
                 (com:invoke-dispatch-method i-controls "playItem"	;(i-media item)
                  (com:invoke-dispatch-method (i-playlist pane player) "item" index)))
               (capi:beep-pane pane)))					; something wrong
          ((com:invoke-dispatch-get-property i-controls "isAvailable" "play")
           (com:invoke-dispatch-method i-controls "play"))
          (t (capi:beep-pane pane)))))

(defmethod vendor-pause ((player wm-player) pane &key (on nil specified-p))
 ;;; See also: wmp-play-state-change
  (let ((i-controls (i-controls pane player)))
    (if specified-p
        (if on
            #2=(if (com:invoke-dispatch-get-property i-controls "isAvailable" "pause")
                   (com:invoke-dispatch-method i-controls "pause")
                   (capi:beep-pane pane))
            #3=(vendor-play player pane))			 	; resume
        (case (com:invoke-dispatch-get-property (i-dispatch pane player) "playState")
          (2 #3#)				 			; toggle Paused
          (3 #2#)))))							; toggle Playing

(defmethod vendor-rewind ((player wm-player) pane)
  (setf (vendor-time player pane) 0d0))

(defmethod vendor-stop ((player wm-player) pane &key silent)
 ;;; Q: After can play again?
  (let ((i-controls (i-controls pane player)))
    (cond ((com:invoke-dispatch-get-property i-controls "isAvailable" #1="stop")
           (com:invoke-dispatch-method i-controls #1#))
          ((not silent)
           (capi:beep-pane pane)))))
        ;; Close the current digital media file, not Windows Media Player itsel
        ;(com:invoke-dispatch-method (i-dispatch pane player) "close"))))

;;; Play reverse or forward at five times the normal speed.
;;; Invoking these methods changes the Settings.rate property to 5.0.
;;; If rate is subsequently changed, or if play or stop is called, player will cease
;;; fast reverse or forward.
;;; TODO: Arbitrary rate

(defmethod vendor-fast-backward  ((player wm-player) pane &key (rate 5.0))
  (declare (ignore rate))
  (let ((i-controls (i-controls pane player)))
    (if (com:invoke-dispatch-get-property i-controls "isAvailable" "FastReverse")
        (com:invoke-dispatch-method i-controls "fastReverse")
        (capi:beep-pane pane))))

(defmethod vendor-fast-forward  ((player wm-player) pane &key (rate 5.0))
  (declare (ignore rate))
  (let ((i-controls (i-controls pane player)))
    (if (com:invoke-dispatch-get-property i-controls "isAvailable" "FastForward")
        (com:invoke-dispatch-method i-controls "fastForward")
        (capi:beep-pane pane))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  EVENTS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. On finishing the entire playlist, player moves to the first item automatically
;;;    even if autoloop is false.
;;;
;;; 2. CurrentItemChange is fired
;;;	- either immediately before OpenStateChange 13 (is "duration" available then?),
;;;	- or after OpenStateChange, 1 PlayStateChange and MediaChange.
;;;    So on-open looks enough and we do not need on-current.
;;;
;;; MS: Player states are not guaranteed to occur in any particular order.
;;;	Furthermore, not every state necessarily occurs during a sequence of events.
;;;	You should not write code that relies upon state order.

;(define-event :wmp "PlaylistChange" wmp-playlist-change)
#+debug
(define-event :wmp "CurrentPlaylistChange" wmp-playlist-change)
#+debug
(defun wmp-playlist-change (pane vector)	;&optional current
 ;;; Handle the event for current playlist only
  ;; BAD: Never fired in delivered.
  (let* ((player (player pane))
         (operation (svref vector 0)))
    #+ys-product (yl:debug-format :media "Playlist change ~s ~s"
                     (com:invoke-dispatch-get-property (i-playlist pane player) "name")
                     operation)
    (case operation
     ;0	Unknown
     ;1		Clear
     ;2		InfoChange
     ;3		Move
     ;(4	; Delete
     ;((5 6)	; Insert Append
       ;(when (and (< (index player) 0)				; no current so far
       ;           (not (vendor-get player pane :autostart)))
       ;  (vendor-set-current-item player pane :index 0)))
     ;8		NameChange
     ;9		Not supported
     ;(10	Sort
) ) )

(define-event :wmp "CurrentItemChange" wmp-current-item-change) 
(defun wmp-current-item-change (pane vector)
 ;;; CAUTION: vector[0] is a COM-interface, which is an ugly simplified:
  ;;		- it is not identical to any i-media returned by "newMedia",
  ;;		- invoking the "duration" property raises an exception.
  (let* ((player (player pane))
         (i-media (svref vector 0))
              ;(com:invoke-dispatch-get-property (i-controls pane player) "currentItem")
              ;(com:invoke-dispatch-get-property (i-dispatch pane player) "currentMedia")
         (file (com:invoke-dispatch-get-property i-media "sourceURL"))
         (item (find-item player :file file))		;;:media i-media
         (items (items player))
         (pos (position item items)))
    #+ys-product
    (yl:debug-format :media "Current item ~s ~s pos ~s~%~t~s"
                     item (or (and item (item-title item))
                              (com:invoke-dispatch-method i-media "getItemInfo" "Title"))
                     pos file)
                     ;(com:invoke-dispatch-method (svref args 0) "isMemberOf"
                     ;                            (i-playlist pane player))
                     ;(com:invoke-dispatch-method i-media "getItemInfo" "TrackingID")
    (when pos
      (setf (slot-value player 'i-current-media) nil
            (index player) pos
            (item-time item) 0)
      (unless (item-duration item)
         (setf (item-duration item) (vendor-duration player pane)))  ; -> i-current-media
      (on-state-change (capi:top-level-interface pane) pane) )))
      ;(on-current (on-state-change (capi:top-level-interface pane) pane :index pos)
    
#+debug
(define-event :wmp "MediaChange" wmp-media-change) 
#+debug
(defun wmp-media-change (pane vector)
 ;;; MediaChange fires too often and is rather boring...
  #-ys-product (declare (ignore pane vector))
  #+ys-product 
  (let ((player (player pane)))
    (multiple-value-bind (item pos) (find-item player :media (svref vector 0))
      (yl:debug-format :media "Media change ~s pos ~s" item pos))))

(define-event :wmp "OpenStateChange" wmp-open-state-change) 
(defun wmp-open-state-change (pane vector)
  ;; NB: Player can go through several open states while it attempts
  ;;	 to open a network file, such as
  ;;	- locating the server,
  ;;	- connecting to the server,
  ;;	- and finally opening the file.
  ;;
  ;; BAD: Never fired in delivered.
  (case (svref vector 0)
    ;0	Undefined		 Player is in an undefined state.
    ;1	PlaylistChanging	New playlist is about to be loaded.
    ;2	PlaylistLocating	Player is attempting to locate the playlist.
    ;				The playlist can be local (library or metafile 
    ;				with an .asx file name extension) or remote.
    ;3	PlaylistConnecting	Connecting to the playlist.
    ;4	PlaylistLoading	Playlist has been found and is now being retrieved.
    ;5	PlaylistOpening	Playlist has been retrieved and is now being parsed and loaded.
    ;6	PlaylistOpenNoMedia	Playlist is open.
    ;7	PlaylistChanged		A new playlist has been assigned to currentPlaylist.
    ;8	MediaChanging		A new media item is about to be loaded.
    ;9	MediaLocating		Windows Media Player is locating the media item.
    ;				The file can be local or remote.
    ;10 MediaConnecting		Connecting to the server that holds the media item.
    ;11 MediaLoading		Media item has been located and is now being retrieved.
    ;12 MediaOpening		Media item has been retrieved and is now being opened.
    (13	;; MediaOpen		Media item is now open.
     (let* ((player (player pane)))
            ;(item (item player)))			; duration is available only now
       ;(if (photo-p item)
       ;    (when-let (duration (item-duration item))
       ;      (setf (vendor-duration player pane (i-media item)) duration))
       (setf (state player) :open
             (status player) nil))
     (on-open (capi:top-level-interface pane) pane))
    ;14 BeginCodecAcquisition	Starting codec acquisition.
    ;15 EndCodecAcquisition	Codec acquisition is complete.
    ;16 BeginLicenseAcquisition Acquiring a license to play DRM protected content.
    ;17 EndLicenseAcquisition	License to play DRM protected content has been acquired.
    ;18 BeginIndividualization	Begin DRM Individualization.
    ;19 EndIndividualization	DRM individualization has been completed.
    ;20 MediaWaiting		Waiting for media item.
    ;(21 OpeningUnknownURL	Opening a URL with an unknown type.
) )

(define-event :wmp "PlayStateChange" wmp-play-state-change) 
(defun wmp-play-state-change (pane vector &optional state)
 ;;; Value: True if handled
  ;; BAD: Never fired in delivered.
  (let ((player (player pane))
        item)
    (case (setf (state player) (or state (vendor-state player pane (svref vector 0))))
      (:play
       ;; Workaround: "PlayStateChange" is not fired in delivered
       (when (setq item (item player))
         (unless (item-duration item)
           (setf (item-duration item) (vendor-duration player pane)))
         (setf (item-time item) (vendor-time player pane)))
       #1=(setf (status player) (vendor-status player pane))
       (on-play (capi:top-level-interface pane) pane)
       t)
      (:pause
       #1#
       #2=(when (setq item (item player))
            (setf (item-time item) (vendor-time player pane)))
       (on-pause (capi:top-level-interface pane) pane)
       t)
      (:end
       #1# #2#
       (on-end (capi:top-level-interface pane) pane)
       t)
      (:stop
       #1# #2#
       (on-stop (capi:top-level-interface pane) pane)		; really stoopped
       t) )))
  ;(shiftf (cdr play-states) (car play-states) new)))

(define-event :wmp "PositionChange" wmp-position-change)
(defun wmp-position-change (pane vector)
 ;;; This event is not raised routinely during playback.
  ;; It only occurs when something actively changes the current position of
  ;; the playing media item, such as
  ;; - the user moving the seek handle or
  ;; - code specifying a value for Controls.currentPosition.
  ;; WMP BAD: Seems no way to permanently monitor the current position during play by code.
  (let (;(old (svref args 0))			; double float
        (new (round (svref vector 1)))		; actually 0 is passed
        (item (item (player pane))))
    (on-state-change (capi:top-level-interface pane) pane
                     :time (if item (setf (item-title item) new) new))))

(define-event :wmp "StatusChange" wmp-status-change)
(defun wmp-status-change (pane vector)
 ;;; Occurs when the "status" property changes value, i.e. the value of
  ;;	(com:invoke-dispatch-get-property (i-dispatch pane) "status")
  ;; NB: Acctually follows every other event.
  (declare (ignore vector))
  (let* ((player (player pane))
         (state (vendor-state player pane)))
    #+(and ys-product debug)
    (when (and yl::*debug-stream* (member :media yl::*debug-current* :test #'eq))
      (princ #\Space yl::*debug-stream*) (princ state yl::*debug-stream*))
    ;; Workaround: "PlayStateChange" is never fired in delivery.
    ;; BAD: "Play" can be invoked from here several times.
    ;;       "Stop" is never invoked but hangs up the application!
    (unless (wmp-play-state-change pane #(0) state)		; pass steate explicitly
      (on-state-change (capi:top-level-interface pane) pane	; state is not handled
                       :state state
                       :status (setf (status player) (vendor-status player pane))
                       :time (vendor-time player pane)))))

(define-event :wmp "Click" wmp-click) 
(defun wmp-click (pane vector)
 ;;; Occurs when the user clicks a mouse button:
  ;;	MouseDown MouseUp Click  or  MouseDown Click [DoubleClick] MouseUp
  ;; Args: vector ::= #(button shiftState X Y)
  ;;	    button
  ;;		Number (int) specifying a bitfield with bits corresponding to
  ;;		bit 0 - left button (value 1),
  ;;		bit 1 - right button (value 2),
  ;;		bit 2 -  middle button (value 4).
  ;;		Only one of the bits is set, indicating the button that caused the event.
  ;;	    shiftState
  ;;		Number (int) specifying a bitfield with the least significant bits:
  ;;		bit 0 - the SHIFT key (value 1)
  ;;		bit 1 - the CTRL key (value 2),
  ;;		bit 2 - and the ALT key (value 4).
  ;;	    x,y
  ;;		Numbers (long) specifying the x or y coordinate of the mouse pointer
  ;;		relative to the upper-left corner of the control.
 ;;; BAD: Never fired in delivered.
  (let (player)
    (cond ((and (eql (svref vector 0) 1)			; left button
                (eql (svref vector 1) 0)			; no shift state
                (< (svref vector 3)				; abouve controls arae
                   (- (capi:simple-pane-visible-height pane)
                      (if (eq (vendor-ui-mode (setq player (player pane)) pane) :none)
                          1 40))))
           (vendor-pause player pane)) )))			; toggle pause/play

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  ERRORS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Set Settings.enableErrorDialogs to false if you choose to display custom error messages

(define-event :wmp "Error" wmp-error)
(defun wmp-error (pane vector)
  (declare (ignore vector))
  (on-error (capi:top-level-interface pane) pane
            "WMP ActiveX error: ~s."
            (let* ((i-error (com:invoke-dispatch-get-property (i-dispatch pane) "error"))
                   (i-item (com:invoke-dispatch-get-property i-error "item" 0)))
              (com:invoke-dispatch-get-property i-item "errorDescription"))))


#||;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
#+unused
(define-vendor :wmp6
               :title "Windows Media Player 6.4"
               :component-name "MediaPlayer.MediaPlayer.1"
               :player-class 'wm-player6
               :sink-callback 'wmp-callback)
#+bad
(defmethod (setf vendor-duration) ((sec number) (player wm-player)
                                   &optional pane (i-media (i-current-media pane player)))
 ;;; WMP: "duration" is a read-only Number (double) and no way to specify photo duration!
  (when i-media
    (com:invoke-dispatch-put-property i-media "duration" (float sec 1d0))))

#+unused
(defun wmp-print-errors (pane)
 ;;; Player can generate a number of errors in response to an error condition.
  ;; The index numbers for the error queue begin with zero.
  (let* ((i-error (com:invoke-dispatch-get-property (i-dispatch pane) "error"))
         (count (com:invoke-dispatch-get-property i-error "errorCount")))
    (format t "Error count ~d~{~%~a~}"
            count
            (loop for i upfrom 0 below count
                  for i-item = (com:invoke-dispatch-get-property i-error "item" i)
                  collect (list (com:invoke-dispatch-get-property i-item "errorCode")
                                (com:invoke-dispatch-get-property i-item
                                                                  "errorDescription")))) ))
#+old
(defun wmp-callback (pane method-name kind vector)
 ;;; An example callback function that responds to the control's events.
  ;; Args: kind ::= :method | :get | :put
  ;;       vector   Simple vector, may be empty
  ;(when (eq kind :method)
  (let ((handler (get-event-handler :wmp method-name)))
    (cond (handler
           #+ys-product
           (unless (equal method-name "StatusChange")
             (yl:debug-format :media "WMP handling event ~s ~s." method-name vector))
           (if (consp handler)
               (apply (first handler) pane vector (rest handler)) ; pass extra args
               (funcall handler pane vector)))			; no extra args
          #+debug
          ((or (equal method-name "MouseMove")			; do not log user gestures
               (equal method-name "MouseDown") (equal method-name "MouseUp")
               (equal method-name "Click") (equal method-name "DoubleClick")
               (equal method-name "KeyDown") (equal method-name "KeyUp")
               (equal method-name "PlaylistChange")
          ))
          #+ys-product
          (t (yl:debug-format :media "WMP unhandled event ~s (~s) ~s."
                              method-name kind vector)) )))
||#

