;;; Ystok-Media - Media player by means of ActiveX (OLE pane) on LispWorks for Windows
;;; Copyright (c) 2025 Dr. Dmitry Ivanov. All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; VLC Player support
;;; https://wiki.videolan.org/Documentation:WebPlugin/

(in-package :ystok.media)

(define-vendor :vlc :title "VLC Player"
               :component-name "VideoLAN.VLCPlugin"
               :player-class 'vlc-player)
               ;:sink-callback 'vlc-callback)

;; As per vlc.input.state, NIL if is unknown or idle.
(defparameter *vlc-states*
 #(:undefined	; 0
   :open	; 1 opening
   :buffer	; 2 bufferring
   :play	; 3 
   :pause	; 4
   :stop	; 5 stopping
   :end		; 6 ended
   :error))	; 7

(defclass vlc-player (player)
 ((vendor-id :allocation :class :initform :vlc)
  ;; Slots holding instances of com:com-interface i-dispatch for invoking
  ;; the OLE properties and methods for inner objects
  (i-input :initform nil)
  (i-video :initform nil)
  ;(i-audio :initform nil)
  ;(i-description :initform nil)				; mediaDescripton
  (mouse-up :initform nil)			; vector of the last MouseUp event
))

;;; I-DISPATCH getters

(defmethod i-playlist (pane (player vlc-player))
  (com:invoke-dispatch-get-property (i-dispatch pane player) "playlist"))

(defmethod i-input (pane &optional (player (player pane)))
  (or (and player (slot-value player 'i-input))
      (let ((i (com:invoke-dispatch-get-property (i-dispatch pane player) "input")))
        (if player (setf (slot-value player 'i-input) i) i))))

(defmethod i-video (pane &optional (player (player pane)))
  (or (and player (slot-value player 'i-video))
      (let ((i (com:invoke-dispatch-get-property (i-dispatch pane player) "video")))
        (if player (setf (slot-value player 'i-video) i) i))))

(defmethod vendor-component-on-insert ((player vlc-player) pane)
 ;;; Rely on the default playlist that seems to be created automatically.
  (let ((plist (CAPI:CAPI-OBJECT-PLIST pane)))
    (setf (vendor-ui-mode player pane :insert t) (getf plist :ui-mode *ui-mode*)))
  ;; Scaling factor as float (supported in vlc version >= 3.0.0).
  ;; Ratio
  ;; - of the number of pixels on screen
  ;; - to the number of pixels in the original decoded video in each dimension.
  ;; Zero is a special value and adjusts the video to the output window..
  (com:invoke-dispatch-put-property (i-video pane player) "scale" 0.0))	; a la strechToFit
  ;(com:invoke-dispatch-put-property (i-dispatch pane player) "branding" 0)

(defmethod vendor-component-on-close ((player vlc-player) pane)
  (dolist (slot-name '(i-input i-video))
    (when-let (i (slot-value player slot-name))
      (com:release i)
      (setf (slot-value player slot-name) nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PLAYLIST, ITEMS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod vendor-add ((player vlc-player) pane 
                       &key pathname title
                            (clear (vendor-get player pane :singleton *singleton*))
                            (photo-duration *photo-duration*))
 ;;; Args: photo-duration
  ;;		 Time in seconds passed as :image-duration option a la
  ;;		 Settings -> Input/codecs / Demusers -> Image
  ;;		 ( -> / ->  -> )
  ;;       other-keys
  ;;       	 Must be passed to COM-method in a string separated by space, e.g.
  ;;       	 ":aspect-ratio=4:3 --rtsp-tcp"
  (let* ((url (string-append "file:///" (namestring pathname)))		; MRL
         (i-playlist (i-playlist pane player))
         (duration (and (photo-p pathname) photo-duration)))
    (when clear
      (com:invoke-dispatch-method i-playlist "stop")
      (vendor-clear player pane))
    (let* ((index (if duration
                      (com:invoke-dispatch-method i-playlist "add" url
                         (pathname-name pathname)		; item name
                         (format nil ":image-duration=~d" photo-duration))
                      (com:invoke-dispatch-method i-playlist "add" url)))
           (item (make-instance 'item
                                 :title (or title (pathname-name pathname))
                                 :url pathname
                                 :duration duration)))
      (declare (ignorable index))
      #+debug (assert (eql (length (items player)) index))
      (push-end item (items player))
      (when (< (index player) 0)				; simplest synchronization
        (setf (index player) 0))
      #+debug (assert (eql (length (items player)) (vendor-items-count player pane) ))
      item)))

(defmethod vendor-remove ((player vlc-player) pane &key index (item (item player index)))
  (cond ((null item)
         nil)
        ((if index 
             (< -1 index (length (items player)))
             (setq index (position item (items player))))
         (let ((pos (com:invoke-dispatch-get-property (i-playlist pane player)
                                                      "currentItem")))
           ;; WORKAROUND: Before remove current single item, ensure stopping it
           (when (and (= pos index 0) 				; single item left
                      (= (vendor-items-count player pane) 1))
             (vendor-stop player pane :silent t))
           (cond ((null (removef (items player) item))		; no item will stay in
                  (setf (index player) -1
                        (state player) nil
                        (status player) nil))
                 ((< index pos)					; removed preceeds current
                  (decf (index player)))			; decrease current index
                 ((= pos index (length (items player)))		; removing current at end
                  (setf (index player) 0)))			; first is current now
           (com:invoke-dispatch-method (i-playlist pane player) "removeItem" index)))
        (t (capi:beep-pane pane))))

(defmethod vendor-clear ((player vlc-player) pane)
  (com:invoke-dispatch-method (i-playlist pane player) "clear"))

(defmethod vendor-items-count ((player vlc-player) pane)
  (com:invoke-dispatch-get-property (i-playlist pane player) "itemCount"))

(defmethod vendor-next ((player vlc-player) pane)
 ;;; Move and start playing regardles the :autostart property
  (com:invoke-dispatch-method (i-playlist pane player) "next"))

(defmethod vendor-previous ((player vlc-player) pane)
 ;;; Move and start playing regardles the :autostart property
  (com:invoke-dispatch-method (i-playlist pane player) "prev"))		; start playingauto

(defmethod vendor-item-descripton ((player vlc-player) pane &key (item (item player)))
 ;;; Extract info form MediaDescription Object
  ;; CAUTION:
  ;;	(com:invoke-dispatch-method (i-video pane player) "description"
  ;;                                 (com:invoke-dispatch-get-property
  ;;                                  (i-video pane player) "track"))
  ;; 	returns video track name, not a MediaDescription object!
  ;;	where track 0 - corresponds to disable,
  ;;	 	    1 - the first video track.
  ;;		    -1 ??
  (when item		; seems only the current item can be asked
   (let* ((i (i-media item))
          (i-description (or i			;(and (or (null item) (current-p item)))?
                             (setf i (com:invoke-dispatch-get-property
                                      (i-dispatch pane player) "mediaDescription")
                                   (i-media item) i))))
    (format nil "Title: ~a~%Artist ~a~%Track ~s~%~a"
            (com:invoke-dispatch-get-property i-description "title")	; namestring
            (com:invoke-dispatch-get-property i-description "artist")
            (com:invoke-dispatch-get-property i-description "trackNumber")
	    (com:invoke-dispatch-get-property i-description "description")))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PROPERTIES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod vendor-get ((player vlc-player) pane (prop (eql :autoloop)) &optional default)
  (com:invoke-dispatch-get-property (i-dispatch pane player) "autoloop"))

(defmethod vendor-put ((player vlc-player) pane (prop (eql :autoloop)) value)
  (com:invoke-dispatch-put-property (i-dispatch pane player) "autoloop" value))

(defmethod vendor-get ((player vlc-player) pane (prop (eql :autostart)) &optional default)
  (com:invoke-dispatch-get-property (i-dispatch pane player) "autoplay"))

(defmethod vendor-put ((player vlc-player) pane (prop (eql :autostart)) value)
  (com:invoke-dispatch-put-property (i-dispatch pane player) "autoplay" value))

(defmethod vendor-get ((player vlc-player) pane (prop (eql :rate)) &optional default)
  (com:invoke-dispatch-get-property (i-input pane player) "rate"))

(defmethod vendor-put ((player vlc-player) pane (prop (eql :rate)) (value number))
  (com:invoke-dispatch-put-property (i-input pane player) "rate" (float value)))

(defmethod vendor-ui-mode ((player vlc-player) pane)
 ;;; UI Mode is defined basing on whether controls are shown and full screen button enabled
  (let ((i-dispatch (i-dispatch pane player)))
    (if (com:invoke-dispatch-get-property i-dispatch "controls")
        (if (com:invoke-dispatch-get-property i-dispatch "fullscreenEnabled") :mini :full)
        :none)))
    
(defmethod (setf vendor-ui-mode) (value (player vlc-player) pane &key insert)
  (declare (ignore insert))
  (let ((i-dispatch (i-dispatch pane player)))
    (case value
      (:none
       (com:invoke-dispatch-put-property i-dispatch "fullscreenEnabled" 0)
       (com:invoke-dispatch-put-property i-dispatch "toolbar" 0)) ;"controls" does not work
      (:mini
       (com:invoke-dispatch-put-property i-dispatch "toolbar" 1)
       (com:invoke-dispatch-put-property i-dispatch "fullscreenEnabled" 0))
      (:full
       (com:invoke-dispatch-put-property i-dispatch "toolbar" 1)
       (com:invoke-dispatch-put-property i-dispatch "fullscreenEnabled" 1)) ))
  value)

(defmethod vendor-state ((player vlc-player)
                         &optional pane (index (com:invoke-dispatch-get-property
                                                (i-input pane player) "state")))
  (when (< -1 index (load-time-value (length *vlc-states*)))
    (svref *vlc-states* index)))

(defmethod vendor-duration ((player vlc-player) &optional pane i-media)
  (declare (ignore i-media))				; available for current item only
  (let ((msec (com:invoke-dispatch-get-property (i-input pane player) "length")))
    (if (and msec (< 0 msec)) (round msec 1000) nil)))	; msec -> sec

(defmethod vendor-time ((player vlc-player) &optional pane)
  (let ((msec (com:invoke-dispatch-get-property (i-input pane player) "time")))
    (if (and msec (<= 0 msec)) (round msec 1000) nil)))

(defmethod (setf vendor-time) ((sec number) (player vlc-player) &optional pane)
  (com:invoke-dispatch-put-property (i-input pane player) "time"
                                    (truncate (* sec 1000))))

(defmethod vendor-position ((player vlc-player) &optional pane)
  (let ((double (com:invoke-dispatch-get-property (i-input pane player) "position")))
    (if (and double (<= 0 double)) double nil)))		; (rationalize 0.012)?

(defmethod (setf vendor-position) ((number float) (player vlc-player) &optional pane)
  (com:invoke-dispatch-put-property (i-input pane player) "position"
                                    (float number 1d0)))	; for sure

(defmethod vendor-version ((player vlc-player) &optional pane)	; "getVersionInfo" signals
  (com:invoke-dispatch-get-property (i-dispatch pane player) "versionInfo"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  METHODS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod vendor-play ((player vlc-player) pane
                        &key item (index (if item (position item (items player)) nil)))
 ;;; TODO: Checking status
  (let ((i-playlist (i-playlist pane player)))
    (if index
        (com:invoke-dispatch-method i-playlist "playItem" index) 
        (com:invoke-dispatch-method i-playlist "play"))))

(defmethod vendor-pause ((player vlc-player) pane &key (on nil specified-p))
  (if specified-p
      (if on
          (com:invoke-dispatch-method (i-playlist pane player) "pause")
          (vendor-play player pane))
      (com:invoke-dispatch-method (i-playlist pane player) "togglePause")))

(defmethod vendor-rewind ((player vlc-player) pane)
  (setf (vendor-position player pane) 0d0))

(defmethod vendor-stop ((player vlc-player) pane &key silent)
  (declare (ignore silent))
  (com:invoke-dispatch-method (i-playlist pane player) "stop"))

(defmethod vendor-fast-forward ((player vlc-player) pane &key rate)
  (setf (vendor-get player pane :rate) (or rate (* (vendor-get player pane :rate) 2.0))))

(defmethod vendor-fast-backward ((player vlc-player) pane &key rate)
  (setf (vendor-get player pane :rate) (or rate (/ (vendor-get player pane :rate) 2.0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  EVENTS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; On finishing the entire playlist, player freezes on the last item
;;; No events indicates fast-forward/backward

(define-event :vlc "MediaPlayerMediaChanged" vlc-media-changed) 
(defun vlc-media-changed (pane vector)
  (declare (ignore vector))
  (let* ((player (player pane))
         (pos (com:invoke-dispatch-get-property (i-playlist pane player) "currentItem")))
    (when (<= 0 pos)
      (when-let (item (item player pos))
        #+(and ys-product debug)
        (let* ((i (i-media item))
               (i-description (or i
                                  (setf i (com:invoke-dispatch-get-property
                                           (i-dispatch pane player) "mediaDescription")
                                        (i-media item) i)))
               (file (item-url item)))
                   ;(com:invoke-dispatch-get-property i-description "URL")))	; => ""
          (yl:debug-format :media "Current item ~s ~s pos ~s~%~t~s"
                           item
                           (or (item-title item)
                               (com:invoke-dispatch-get-property i-description "title"))
                           pos file))
        (setf (item-time item) 0
              (item-position item) 0d0
              (index player) pos)))))
      ;(on-current (capi:top-level-interface pane) pane :index pos)

(define-event :vlc "MediaPlayerOpening" vlc-opening)
(defun vlc-opening (pane vector)
 ;;; Q: Should we rely on "MediaPlayerMediaChanged" instead?
  (declare (ignore vector))
  (let ((player (player pane)))
    (setf (state player) :open
          (status player) nil))
  (on-open (capi:top-level-interface pane) pane))

(define-event :vlc "MediaPlayerPlaying" vlc-playing) 
(defun vlc-playing (pane vector)
 ;;; Usually fired before Buffing or after Paused
  (declare (ignore vector))
  (let* ((player (player pane))
         (item (item player)))
    (when item
      ;; Workaround: "MediaPlayerLengthChanged" is not fired in delivered
      (unless (item-duration item)
        (setf (item-duration item) (vendor-duration player pane)))
      ;; Workaround: "MediaPlayerTimeChanged" is never fired in delivered.
      (setf (item-time item) (vendor-time player pane)))
    (setf (state player) :play))
  (on-play (capi:top-level-interface pane) pane))

(define-event :vlc "MediaPlayerPaused" vlc-pause) 
(defun vlc-pause (pane vector)
  (declare (ignore vector))
  (let ((player (player pane)))
    ;; Workaround: "MediaPlayerTimeChanged" is never fired in delivered.
    (when-let (item (item player))
      (setf (item-time item) (vendor-time player pane)))
    (setf (state player) :pause))
  (on-pause (capi:top-level-interface pane) pane))

(define-event :vlc "MediaPlayerEndReached" vlc-end-reached) 
(defun vlc-end-reached (pane vector)
  (declare (ignore vector))
  (setf (state (player pane)) :end)
  (on-end (capi:top-level-interface pane) pane))

(define-event :vlc "MediaPlayerStopped" vlc-stopped)
(defun vlc-stopped (pane vector)
 ;;; Called by: Even just after launching and adding a new item.
  ;; NB:     
  (declare (ignore vector))
  (setf (state (player pane)) :stop)
  (on-stop (capi:top-level-interface pane) pane))

(define-event :vlc "MediaPlayerLengthChanged" vlc-length-changed) 
(defun  vlc-length-changed (pane vector)
 ;;; Ussually fired just before playing
 ;;; BAD: Never fired in delivered.
  (let ((duration (round (svref vector 0) 1000))
        (player (player pane)))
    (when-let (item (item player))
      (setf (item-duration item) duration))
    (on-state-change (capi:top-level-interface pane) pane :duration duration)))

(define-event :vlc "MediaPlayerTimeChanged" vlc-time-changed) 
(defun vlc-time-changed (pane args)
 ;;; BAD: Never fired in delivered.
  (let* ((player (player pane))
         (item (item player))
         (sec (round (svref args 0) 1000))
         (position (vendor-position player pane)))
    (on-state-change (capi:top-level-interface pane) pane
                     :time (if item (setf (item-time item) sec) sec)
                     :position (if item (setf (item-position item) position) position) )))

;(define-event :vlc "MediaPlayerPositionChanged" vlc-position-changed) 
;(defun vlc-position-changed (pane args)
;;; Do we need the additional call of on-state-change? Let's rely on time-changed only.
;  (on-state-change (capi:top-level-interface pane) pane :position (svref args 0)))

(define-event :vlc "MouseUp" vlc-mouse-up) 
(defun vlc-mouse-up (pane vector)
 ;;; Args: See wmp-click
  ;; NB: Is not fired when the user clicks on the controls area
  ;; BAD: Never fired in delivered so toggle play/pause does not work.
  (let ((player (player pane)))
    (setf (slot-value player 'mouse-up)
          (if (and (eql (svref vector 0) 1)			; left button
                   (eql (svref vector 1) 0))			; no shift state
              (cons (svref vector 2) (svref vector 3))		; (x . y)
              nil))))						; drop coordinates

(define-event :vlc "Click" vlc-click) 
(defun vlc-click (pane vector)
 ;;; Occurs when the user clicks a mouse button:
  ;;	MouseDown MouseUp Click
  ;;	MouseDown MouseUp Click DblClick MouseUp Click
  ;; Args: vector #()
  ;; CAUTION: DblClick turns the window to full screen only when the following is true:
  ;;		(com:invoke-dispatch-get-property i-dispatch "fullScreenEnabled")
  ;; TODO: Ignore toggling pause/play just after DblClick.
 ;;; BAD: Never fired in delivered.
  (declare (ignore vector))
  (let ((player (player pane)))
    (cond ((slot-value player 'mouse-up)
           (vendor-pause player pane)) )))			; toggle pause/play

#|;;; USUAL EVENT SEQUENCE
 "MediaPlayerMediaChanged" #()
 "MediaPlayerOpening" #()
 "MediaPlayerBuffering" #(0)
 "MediaPlayerSeekableChanged" #(T)
 "MediaPlayerPausableChanged" #(T)
 "MediaPlayerLengthChanged" #(5000).
 "MediaPlayerPlaying" #().
 "MediaPlayerBuffering" #(0) ... "MediaPlayerBuffering" #(100)
 "MediaPlayerPositionChanged" #(0.1) "MediaPlayerTimeChanged" #(405)
 ...
 "MediaPlayerPositionChanged" #(0.9) "MediaPlayerTimeChanged" #(4600)
 "MediaPlayerEndReached" #()
 "MediaPlayerStopped" #()
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  ERRORS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-event :vlc "MediaPlayerEncounteredError" vlc-encountered-error)
(defun vlc-encountered-error (pane args)
 ;;; VLC BAD: No way to extract error string
  (on-error (capi:top-level-interface pane) pane
            "VLC ActiveX error: ~s." args))
 ;(capi:report-active-component-failure pane (vendor-component-name (get-vendor :vlc))
 ;                                      :unknown :unknown args))


#||;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  NOT WORKING ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod vendor-background ((player vlc-player) &optional pane)
  (integer-rgb-color (com:invoke-dispatch-get-property (i-dispatch pane player)
                                                       "bgcolor")))	; does not work
 
(defmethod (setf vendor-background) (value (player vlc-player) &optional pane)
  (let ((color (if (keywordp value) (color:get-color-spec value) value)))
    (com:invoke-dispatch-put-property (i-dispatch pane player) "bgcolor"
                                      (rgb-integer (color:color-red color)
                                                   (color:color-blue color)
                                                   (color:color-green color)))))

(defmethod vendor-fast-backward ((player vlc-player) pane &key (rate -4.0))
 ;;; Negative rate seems treated as 1.0
  (setf (vendor-get player pane :rate) rate))

;;; Args: pathname, text  Non-NIL - enable, NIL - disable.

(defmethod vendor-set-logo ((player vlc-player) pathname &optional pane)
 ;;; May be logo is shown between itmes in a playlist?
  (let ((i-logo (com:invoke-dispatch-get-property (i-video pane player) "logo")))
    (if pathname
        (progn (com:invoke-dispatch-method i-logo "file" (namestring pathname))
          (com:invoke-dispatch-put-property i-logo "opacity" 255)	; completely opaque
          (com:invoke-dispatch-put-property i-logo "position" "center")
          (com:invoke-dispatch-method i-logo "enable"))
        (com:invoke-dispatch-method i-logo "disable"))))

(defmethod vendor-set-marquee ((player vlc-player) text &optional pane)
  (let ((i-marquee (com:invoke-dispatch-get-property (i-video pane player) "marquee")))
    (if text
        (progn (com:invoke-dispatch-put-property i-marquee "text" text)
          (com:invoke-dispatch-put-property i-marquee "opacity" 255)	; completely opaque
          (com:invoke-dispatch-put-property i-marquee "color" #xFF0000) ; red
          (com:invoke-dispatch-put-property i-marquee "position" "center")
          (com:invoke-dispatch-put-property i-marquee "size" 20)
          (com:invoke-dispatch-method i-marquee "enable"))
        (com:invoke-dispatch-method i-marquee "disable"))))

#+old
(defun vlc-callback (pane method-name kind vector)
 ;;; An example callback function that responds to the control's events.
  ;; Args: kind ::= :method | :get | :put (the last two seems never passed by VLC.)
  ;;       vector   Simple vector, may be empty
  ;; The code here just keeps
  ;; - the title of the interface,
  ;(when (eq kind :method)
  (let ((handler (get-event-handler :vlc method-name)))
    (cond (handler
           #+ys-product
           (unless (equal method-name "MediaPlayerTimeChanged")
             (yl:debug-format :media "VLC handling event ~s ~s." method-name vector))
           (if (consp handler)
               (apply (first handler) pane vector (rest handler)) ; pass extra args
               (funcall handler pane vector)))			; no extra args
          #+debug
          ((or (equal method-name "MouseMove")			; do not log user gestures
               (equal method-name "MouseDown") (equal method-name "MouseUp")
               (equal method-name "Click")
               (equal method-name "KeyDown") (equal method-name "KeyUp")
               (equal method-name "MediaPlayerPositionChanged")
          ))
          #+ys-product
          (t (yl:debug-format :media "VLC unhandled event ~s (~s) ~s."
                              method-name kind vector)) )))
|#
