;;; Ystok-Media - Media player by means of ActiveX (OLE pane) on LispWorks for Windows
;;; Copyright (c) 2025 Dr. Dmitry Ivanov. All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Example CAPI interface for experiment

(in-package :cl-user)

(defconstant +media-player-title+ "Ystok Media Player")

;; Map: property -> variable symbol
(defparameter *flag-alist* '((:autoloop  . med:*autoloop*)
                             (:autostart . med:*autostart*)
                             (:singleton . med:*singleton*)))

(capi:define-interface media-player-face ()
 () ;(session :initform nil))
 (:panes
  (current-vendor capi:option-pane
   :title "Active component vendor:"
   :items (cons nil med:*vendors*)
   :data-function (lambda (arg) (if (med::vendor-p arg) (med:vendor-id arg) arg))
   :print-function (lambda (arg) (cond ((med::vendor-p arg) (med:vendor-title arg))
                                       ((keywordp arg) (if (setq arg (med:find-vendor arg))
                                                           (med:vendor-title arg)
                                                           ""))
                                        (t "")))
  ;:selection (vendor-option-selection med:*current-vendor-id*)	; :initial selection
   :keep-selection-p t
   :selection-callback 'vendor-option-on-select
   :callback-type :interface-data)
  (ui-mode capi:option-pane
   :title "Controls over video:"
   :items med:*ui-mode-alist*
   :data-function #'car
   :print-function #'cdr
   :keep-selection-p t
   :selected-item (assoc med:*ui-mode* med:*ui-mode-alist*)
   :selection-callback 'ui-mode-on-select
   :callback-type :interface-data
   :help-key "Anount of embedded controls provided by the vendor component.")
  (photo-duration-sec capi:text-input-range
   :title "Photo durattion, sec:"
   :title-position :left
   :start 1 :end 10
   :value med::*photo-duration*
   :help-key "Playing static graphic files duration in seconds.")
  (flags capi:check-button-panel
   :items (mapcar #'car *flag-alist*)
   :print-function 'string-downcase
   :layout-class 'capi:column-layout
   :selection-callback (lambda (face prop) (set-flag face prop t))
   :retract-callback   (lambda (face prop) (set-flag face prop nil))
   :callback-type :interface-data
   :help-keys '("Whether the current media item begins playing automatically."
                "Duration of static graphic ifles play in seconds."))
  (media-pane med:player-pane
   :visible-border t)
  (controls capi:push-button-panel
   :items `(:play :pause :rewind :stop  :backward :forward  :previous :next)
   :print-function 'string-capitalize
   :callbacks '#1=(operate-on-interface . #1#)
   :callback-type :interface-data
   :help-keys '("Start or contunue playing the current media item."
                "Pause playing."
                "Restart playing the current media item."
                "Stop playing the playlist."
                "Fast playback backward,"
                "Fast playback forwar."
                "Move to previous media item."
                "Move to next media item.")) )

 (:layouts
  (main-layout capi:column-layout '(top-row
                                    media-pane
                                    controls)
   :adjust :center)
  (top-row capi:row-layout '(top-grid flags)
   :gap 10)
  (top-grid capi:grid-layout '(current-vendor	photo-duration-sec
                               ui-mode		nil)
   :x-gap 10
   :visible-max-height t))
 (:menu-bar file-menu)
 (:menus
  (file-menu "Playlist"
   (("Add item..."    :data :open  :accelerator "Control-o")
    ("Remove item..." :data :remove :help-key "Remove an item asked from the playlist.")
    ("Clear"     :data :clear  :help-key "Remove all items from the playlist.")
    ("About..."  :data :about  :help-key "Show version and media item info in dialog."))
   :enabled-function (lambda (face) (med::player (slot-value face 'media-pane)))
   :callback 'operate-on-interface
   :callback-type :interface-data))

 (:default-initargs
  :best-width 700  :best-height 520
  :title +media-player-title+
  :message-area t
  :create-callback (lambda (face)
                     (with-slots (media-pane flags) face
                       (setf (capi:choice-selection flags)
                             (loop for (prop . symbol) in *flag-alist*
                                   and pos upfrom 0
                                   when (med:vendor-get nil media-pane prop
                                                        (symbol-value symbol))
                                   collect pos))))
  #+ys-product #+ys-product
  :help-callback 'ywi:help-callback))

(defun vendor-option-selection (id)
  (let ((pos (position id med:*vendors* :key #'med:vendor-id))) ;med:*vendor-alist*)))
    (if pos (1+ pos) 0)))					; 0 means select nil

(defun vendor-option-on-select (face id)
  (setf (capi:titled-object-message face)
        (format nil "~:[Closing component~;Inserting ~:*~a comopnent~]..." id))
  (med:set-pane-vendor (slot-value face 'media-pane) :id id)
  (capi:redisplay-menu-bar face)
  #+ys-product (setf (ywi:enabled (slot-value face 'controls)) id))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PROPERTIES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ui-mode-on-select (face data)
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (setq med:*ui-mode* data)			; for storing in preferences
    (when player
      (setf (med:vendor-ui-mode player pane) data))))

(defun set-flag (face prop value)
  (let ((pane (slot-value face 'media-pane)))
    (lw:when-let (player (med:player pane))
      (setf (med:vendor-get player pane prop) value))
    (setf (capi:capi-object-property pane prop) value		    ; set pane default
          (symbol-value (sys:cdr-assoc prop *flag-alist*)) value))) ; set global default?

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  METHODS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod operate-on-interface (face (op (eql :open)))
  (with-slots (media-pane photo-duration-sec) face
    (let* ((player (med:player media-pane))
           (filters (med:media-file-filters :flash t)))
      (if player
          (multiple-value-bind (pathname successp)
              (capi:prompt-for-file "Select mediafile to play"
                                    :filters (append filters
                                                     '("All files" "*.*"))
                                    :filter (second filters)
                                    :owner face)
            (when successp
              (med:vendor-add player media-pane 
                  :pathname pathname
                  :photo-duration (capi:text-input-range-value photo-duration-sec))))
          (progn (capi:beep-pane face)
            (setf (capi:titled-object-title face) "Component vendor is not specified"))))))

(defmethod operate-on-interface (face (op (eql :remove)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane))
         (index (med::index player)))
    (when-let (remove (capi:prompt-for-integer "Enter item index to remove:"
                           :min 0
                           :max (1- (med:vendor-items-count player pane))
                           :initial-value (if (<= 0 index) index 0)))
      (med:vendor-remove player pane :index remove))))

(defmethod operate-on-interface (face (op (eql :clear)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-clear player pane)))

(defmethod operate-on-interface (face (op (eql :about)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (capi:display-message "Ystok Media Player: library test application~@
			~@[~a ~]ActiveX version ~a~@
			~%Playlist item count ~s~@
			~%Time ~a/~a, position ~f rate ~f~@
			~%Current Item [~d] Description~@[~%~a~]"
                          (when player (med::vendor-title (med::vendor player)))
                          (med:vendor-version player pane)
                          (med::vendor-items-count player pane)
                          (med:vendor-timestring player pane)
                          (med:vendor-duration-string player pane)
                          (med:vendor-position player pane)
                          (med:vendor-get player pane :rate)
                          (med::index player)
                          (med:vendor-item-descripton player pane))))

(defmethod operate-on-interface (face (op (eql :play)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-play player pane)))

(defmethod operate-on-interface (face (op (eql :pause)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-pause player pane :on t)))

(defmethod operate-on-interface (face (op (eql :rewind)))
  (let ((pane (slot-value face 'media-pane)))
    (med::vendor-rewind (med:player pane) pane)))

(defmethod operate-on-interface (face (op (eql :stop)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-stop player pane)))

(defmethod operate-on-interface (face (op (eql :backward)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-fast-backward player pane)))

(defmethod operate-on-interface (face (op (eql :forward)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-fast-forward player pane)))

(defmethod operate-on-interface (face (op (eql :next)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-next player pane)))

(defmethod operate-on-interface (face (op (eql :previous)))
  (let* ((pane (slot-value face 'media-pane))
         (player (med:player pane)))
    (med:vendor-previous player pane)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  EVENTS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun format-status (player &key (item (med:item player))
                                  (state (slot-value player 'med::state))
                                  (status (med::status player))
                                  time position duration)
 ;;; Helper: Compose message area string
  (format nil "Time ~a/~a~@[ Position ~f~]~@[ ~a~]"
          (when (or time
                    (and item (setq time (med::item-time item))))
            (med:timestring time))
          (when (or duration
                    (and item (setq duration (med::item-duration item))))
            (med:timestring duration))
          (or position
              (and item (med::item-position item)))
          (or status
              (and state (string-capitalize state))) ))

(defmethod med:on-open ((face media-player-face) (pane med:player-pane) &optional item)
  (let* ((player (med:player pane))
         (item (or item (med:item player))))
    (setf (capi:titled-object-title face)
           (format nil "~a - ~a."
                   (if item (or (med::item-title item) (med::item-url item)) nil)
                   +media-player-title+)
          (capi:titled-object-message face)
           (format-status player :item item
                          :status (or (med::status player) "Item has opened")))))
                          ;:time 0 
                          ;:position 0d0
                          ;:duration (med:vendor-duration player pane)))))

(defmethod med:on-play ((face media-player-face) (pane med:player-pane))
  (let ((player (med:player pane)))
    (setf (capi:titled-object-message face)
          (format-status player :status (or (med:status player) "Playing.")))))

(defmethod med:on-pause ((face media-player-face) (pane med:player-pane))
  (let ((player (med:player pane)))
    (setf (capi:titled-object-message face)
          (format-status player :status (or (med:status player) "Paused.")))))

(defmethod med:on-end ((face media-player-face) (pane med:player-pane))
  (let ((player (med:player pane)))
    (setf (capi:titled-object-message face)
          (format-status player :status (or (med:status player) "Ended.")))))

(defmethod med:on-stop ((face media-player-face) (pane med:player-pane))
  (let ((player (med:player pane)))
    (setf (capi:titled-object-title face) +media-player-title+
          (capi:titled-object-message face) (or (med:status player) "Stopped."))))

(defmethod med:on-state-change ((face media-player-face) (pane med:player-pane)
                                &key state status time duration position)
  (let* ((player (med:player pane))
         (item (med:item player)))				; current media item
    (setf (capi:titled-object-message face)
          (format-status player :item item
                         :state (or state (slot-value player 'med:state))
                         :status (or status (med:status player))
                         :time time :position position :duration duration))))

(defun main ()
 ;;; Application command line:
  ;;	yplayer.exe [-log]
  ;; where "-log" turns on a detailed output to the console window.
  (win32:set-application-themed nil)
  #+(and ys-product debug)
  (when (search "log" (second sys:*line-arguments-list*) :test #'char-equal)
    (yl:debug-on :media)
    (setq yl::*debug-stream* *terminal-io*)		; redirect to console
    (yl:debug-format :media "DEBUG STATUS")
    (yl::debug-status))
  (capi:display (make-instance 'media-player-face)))


#||;;;;;;;;;;;;;;;;;;;;;;;;;;;;; EVALUATION ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(capi:display (make-instance 'media-player-face))

(yl:define-debug-option :media :all "Media player dubugging")
(yl:debug-on :media)
(yl:debug-off :media)
(setq yl::*debug-stream* *standard-output*)		; redirect to IDE output tab

(setq f (capi:locate-interface 'media-player-face)
      pane (slot-value f 'media-pane)
      p (med:player pane))

(med:vendor-ui-mode p pane)
(med:vendor-get p pane :autoloop)
(med:vendor-get p pane :autostart)

(med::vendor-state p pane)
(capi:execute-with-interface f 'med:vendor-play p pane)
(capi:execute-with-interface f 'med:vendor-play p pane :index 2)

;;; VLC

(capi:execute-with-interface f 'med::vendor-set-logo
                             p #P"D:/Projects/common/images/ys122x50-web-tran.png" pane)

(med:vendor-background p pane)
(capi:execute-with-interface f #'(setf med:vendor-background) :red p pane)

(capi:execute-with-interface f 'med::vendor-set-marquee p "Long long marquee" pane)
(capi:execute-with-interface f 'med::vendor-set-marquee p nil pane)

(com:invoke-dispatch-get-property (med::i-dispatch pane p) "fullScreenEnabled")
(com:invoke-dispatch-get-property (med::i-video pane p) "track") ; => -1
(com:invoke-dispatch-put-property (med::i-video pane p) "scale" 0.0)

;;; WMP

(med::i-current-media pane p)
(med::vendor-assign-current-item p pane 0)
(setq pl (com:invoke-dispatch-get-property (med::i-dispatch pane p) "currentPlaylist"))
(com:invoke-dispatch-get-property pl "name")
(com:invoke-dispatch-get-property (med::i-playlist pane p) "name")
(med::vendor-i-eq p (med::i-playlist pane p) pl)

(com:invoke-dispatch-method (med::i-settings pane p) "getMode" "loop")
(com:invoke-dispatch-method (med::i-settings pane p) "setMode" "loop" t)

(com:invoke-dispatch-get-property (med::i-dispatch pane p) "stretchToFit")
(com:invoke-dispatch-get-property (med::i-settings pane p) "autoStart")

(com:invoke-dispatch-put-property (med::i-dispatch pane p) "currentMedia" :nothing)

(setq m1 (com:invoke-dispatch-get-property (med::i-dispatch pane p) "currentMedia"))
(setq m2 (com:invoke-dispatch-get-property (med::i-dispatch pane p) "currentMedia"))
(com:invoke-dispatch-method m1 "isIdentical" m2) ; => T

(med::wmp-print-errors pane)

#+old
(defmethod med:on-time-change ((face media-player-face) (pane med:player-pane)
                               &optional sec)
  (let* ((player (med:player pane))
         (item (med::item player)))
    (unless sec
      (setq sec (med:vendor-time (med:player pane) pane)))
    (setf (capi:titled-object-message face)
          (format-status player :item item
                         :time (if item (setf (med::item-time item) sec) sec)))))
#+old
(defmethod med:on-position-change ((face media-player-face) (pane med:player-pane) double)
  (let* ((player (med:player pane))
         (item (med::item player)))
    (setf (capi:titled-object-message face)
          (format-status player :item item
                         :position (if item
                                       (setf (med::item-position item) double)
                                       double)))))
||#
