;;; Ystok-Media - Media player by means of ActiveX (OLE) on LispWorks for Windows
;;; Copyright (c) 2025 Dr. Dmitry Ivanov. All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Specials, generics, and classes

(in-package :ystok.media)

(defparameter *video-file-types*
 '(;"3g2"
   "3gp" ;"asf" "asx"
   "avi" "divx" "drc" "dvb"			;"evo"
   "f4p" "f4v" "flv" "h264" "h265" "hevc"	;"m2ts" "m2v"
   "mkv" "mov" "mp4"
   "mpg" "mpeg" ;"mts" "mxf"			;"ogm"
   "ogv" "qt"					;"rm" "rmvb"
   "ts" "vob" "webm" "wmv"))

(defparameter *flash-file-types* '("flv" "swf"))

(defparameter *audio-file-types*
 '("aac" "ac3"					;"aiff"
   "amr" "ape" "dts" "f4a" "f4b" "flac" "gsm"
   "m4a" "midi"					;"mlp" "mka" "mp2"
   "mp3" "oga" "ogg" "opus"		;"pva" "ra" "ram" "raw" "rf64" "spx" "tta"
   "wav" "wavpack" "wma" "wv"))

(defparameter *photo-file-types* '("bmp" "gif" "jpeg" "jpg" "png"))	; photos

(defun media-file-filters (&key (video t) (audio t) (photo t) (flash nil))
  (append (when video
            (list "Video files" (format nil #1="~{*.~a~^; ~}" *video-file-types*)))
          (when flash
            (list "Flash files" (format nil #1# *flash-file-types*)))
          (when audio
            (list "Audio files" (format nil #1# *audio-file-types*)))
          (when photo
            (list "Photo files" (format nil #1# *photo-file-types*)))))

(defparameter *photo-duration* 5)	; slide duration in seconds (works for VLC only)

(defgeneric photo-p (url)
 (:method (dummy)
  nil)
 (:method ((url pathname))
  (member (pathname-type url) *photo-file-types* :test #'string-equal)))

;;; PREFERENCES

(defparameter *ui-mode-alist* '((:none . "none controls")
                                (:mini . "minimal controls")
                                (:full . "full contorls")))
(defvar *ui-mode* :mini)

(defvar *autoloop* nil)		; loop when reaching the end of playlist
(defvar *autostart* nil)	; start play automatically on assigning current item
(defvar *singleton* nil)	; only one item in playlist

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  VENDOR  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ActiveX control implementation specific parameters

(defstruct (vendor (:type vector) (:copier nil))
  id				; keyword
  title				; string
  component-name		; ProgID or VersionIndependentProgID in Windows Registry
  player-class			; symbol
  sink-callback			; non-standard callback function for default COM-interface
  (events (make-hash-table :test #'equal)))	; mapping: medhod-name -> handler

(defun vendor-p (arg)
  (simple-vector-p arg))

(defvar *vendors* ())			; list of all implemented vendors

(defvar *current-vendor-id* nil)	; keyword

(defmacro define-vendor (vendor-id &rest args)
 `(let ((rest (member ,vendor-id *vendors* ,:key #'vendor-id))
        (vendor (make-vendor ,:id ,vendor-id ,@args)))
    (if rest
        (rplaca rest vendor)
        (push-end vendor *vendors*))
    vendor))

(defmacro find-vendor (id)
  `(first (member ,id *vendors* ,:key #'vendor-id ,:test #'eq)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PLAYER  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Superclass of all vendor-specific players
;;; Each subclass must define vendor-id slot with :allocation :class of its own.

(defclass player ()
 ((i-dispatch :initform nil)		; com:com-interface i-dispatch of a root object
  ;; com:com-interface i-dispatch of a COM-object
  ;; Siimalar to: vlc.playlist or wmp.currentPlaylist
  (i-playlist :initform nil)
  ;; List of item instances, which must be in sync with playlist items COM-collection.
  ;; (e.g. vls.playlist.items).
  ;; NB: For some vendors, such a collection does not exist at all but only through index.
  (items :accessor items :initform ())
  ;; Current item index
  ;; Similar usage: wmp.currentPlaylist.item(index)
  (index :accessor index :initform -1)
  ;; Cached current play/open state as a keyword <= VLC.input.state or WMP.playState
  ;; Possible values are from
  ;;	(union (coerce *vlc-states* 'list) (coerce *wmp-play-states* 'list))
  (state :writer (setf state) :initform nil)
  ;; String string <= wmp.status or NIL
  (status :accessor status :initform nil) ))

(defmethod print-object ((self player) stream)
  (print-unreadable-object (self stream :type nil :identity nil)
    (format stream "~a~@[ ~s~]" (type-of self) (slot-value self 'i-dispatch))))

(defun vendor (player)
 ;;; Corresponding vendor structure object
  (find-vendor (slot-value player 'vendor-id)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  ITEM  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Vendor-independent playlist item
;;; Holds media properties stored initially and last known status properties
;;; A la wmp Media object <= wmp.currentPlaylist.item(index)

(defclass item ()
 (; String usually pathname-name
  ; Alse can be get via VLC.mediaDescription.title or WMP getItemInfo("Title")
  (title :accessor item-title :initarg :title :initform nil)
  ;; Absolute pathname equals to VLC.mediaDescription.URL or WMP.URL
  ;; TODO: uri:uri
  (url :accessor item-url :initarg :url :initform nil)
  ;; In sec <= VLC.input.length or WMP.getItemInfo("Duration")
  (duration :accessor item-duration :initarg :duration :initform nil)
  ; Integer seconds <= VLC.input.time or WMP.currentPosition
  (time :accessor item-time :initform 0)
  ;; String string <= wmp.status or NIL
  ;(status :accessor item-status :initform nil)
  ;; VLC ONLY: Current position as double float <= vlsc.input.position
  (position :accessor item-position :initform nil)
  ;; COM-interface of
  ;; - mediaDescriptionn object for VLC  (does it work if the item is not current?)
  ;; - media object <= WMP.newMedia
  (i-media :accessor i-media :initarg :media :initform nil) ))

(defmethod print-object ((self item) stream)
  (print-unreadable-object (self stream :type nil :identity nil)
    (format stream "~@[~s ~]~a"
            (item-title self) 
            (file-namestring (item-url self)))))

(defmethod photo-p ((item item))
  (photo-p (item-url item)))

(defun item (player &optional (index (index player)))
 ;;; Playlist item acccessor
  ;; Args: index Integer element index (can be out of range).
  ;; Value: NIL if index is out of range
  (if (and index (<= 0 index))
      (nth index (items player))
      nil))

(defun release-item (item)
 ;;; MUST BE CALLED before removing the item instance from (items player)!
  ;; DANGEROUS: Declrese reference count
  (when-let (i (i-media item))
    (com:release i)))

(defgeneric vendor-i-eq (player x y)
 ;;; Value: True if two COM-interfaces x and y reference the same com-object
  ;; Args: x	Usually new argument passed to an event handler.
  ;; 	   y	Usually com-interface of a known type (e.g. media or playlist)
  ;;		stored within a player or item instance.
 (:method (player x y)
  nil)
 (:method (player (x com:com-interface) (y com:com-interface))
  (fli:pointer-eq x y)) )

(defun find-item (player &key media file)
 ;;; Args: media COM-interface i-media
  ;;       url   Sring of full namestring.
  ;; Values: 1) item or NIL.
  ;; 	     2) position in items.
  ;(flet ((%test (x y) (vendor-i-eq player x y)))
   (loop for item in (items player)
         and pos upfrom 0
         when (cond (media (vendor-i-eq player (i-media item) media))
                    (file  (string= (namestring (item-url item)) file)))
         return (values item pos)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PLAYER-PANE  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defclass player-pane (capi:ole-control-pane)
 ;;; ActiveX control pane class
 ((player :accessor player :initarg :player :initform nil) 	; to be set later
  (sink :initarg :sink :initform nil))			; capi:ole-control-pane-simple-sink
 (:default-initargs
  :insert-callback (lambda (pane)				; create player instance
                     (when *current-vendor-id*
                       (when-let (vendor (find-vendor *current-vendor-id*))
                         (vendor-component-on-insert
                          (setf (player pane) (make-instance (vendor-player-class vendor)))
                          pane))))
  :close-callback  (lambda (pane)				; drop player instance
                     (vendor-component-on-close (shiftf (player pane) nil) pane))
))

(defun i-dispatch (pane &optional (player (player pane)))
 ;;; Root object i-dispatch
  ;; CAUTOIN: Unless player instance is kept, a new I-DISPATCH foreign object is created!
  (or (and player (slot-value player 'i-dispatch))
      (let ((i-dispatch (capi:ole-control-i-dispatch pane)))
        (if player
            (setf (slot-value player 'i-dispatch) i-dispatch)	; memoize i-dispatch
            i-dispatch))))

(defun set-pane-vendor (pane &key id (vendor (if (vendor-p id) id (find-vendor id))))
 ;;; Helper: React on changing vendor by the user or programmatically
  ;; Args: id/vendor  ::= non-NIL - insert,
  ;; 			| NIL - close.
  ;; NB: Assign *current-vendor-id* only on inserting/closing or successfully,
  (if vendor
      (let ((sink-callback (or (vendor-sink-callback vendor) 'sink-callback)))
        (setf *current-vendor-id* (vendor-id vendor)
              (capi:component-name pane) (vendor-component-name vendor)	; -> on-insert
              (slot-value pane 'sink) (capi:attach-simple-sink sink-callback
                                                               pane :default)))
      (let ((sink (slot-value pane 'sink)))
        (when sink
          (capi:detach-simple-sink sink pane)
          (setf (slot-value pane 'sink) nil))
        ;; No harm if called when no component inserted
        (capi:ole-control-close-object pane)	 ; -> (setf (capi:component-name ...) nil)
        (setf *current-vendor-id* nil) )))

;;; IMPLEMENTATION

(defun %not-implemented (player method &rest args)
 ;;; Helper
  (capi:display-message "Method ~a~@[ ~s~] is not implemented for~% ~s."
                        method args player))

(defgeneric vendor-component-on-insert (player pane)
 (:method (player pane))

 (:method :before (player pane)
  #+ys-product (yl:debug-format :media "Switching to ~s..." player)
  (let ((plist (CAPI:CAPI-OBJECT-PLIST pane)))
    (vendor-put player pane :autoloop (getf plist :autoloop *autoloop*))
    (vendor-put player pane :autostart (getf plist :autostart *autostart*))))

 #+(and ys-product debug)
 (:method :after (player pane)
  (yl:debug-format :media "Inserted OLE component ~s" player)
  (when (and yl::*debug-stream* (member :media yl::*debug-current* :test #'eq))
    (print-event-hanlers (vendor player) yl::*debug-stream*))) )

(defgeneric vendor-component-on-close (player pane)
 ;;; Release allocated component resources in any
  ;; Called by: capi:ole-control-close-object (and likely close the interface window).
 (:method (player pane))

 (:method :before (player pane)
   #+ys-product (yl:debug-format :media "Closing OLE component ~s" player)
   (when player
     ;(when (item player) (vendor-stop player pane))
     (vendor-clear player pane)))

 (:method :after (player pane)
  (when player
    (when-let (i (slot-value player 'i-playlist))
      (com:release i)
      (setf (slot-value player 'i-playlist) nil)))) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PLAYLIST, ITEMS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defgeneric i-playlist (pane player)
 ;;; Value: Single or current playlist com:com-interface i-dispatch
 (:method :around (pane player)
  (or (slot-value player 'i-playlist)
      (let ((i (call-next-method)))
        (if (and i (not (eq i :nothing)))
            (setf (slot-value player 'i-playlist) i)		; memoize
            nil))))

 (:method (pane player)
  (%not-implemented player 'i-playlist)) )

(defgeneric vendor-items-count (player pane)
 ;;; Number of media items in the current playlist
 (:method (player pane)
  ;; The defautl method relies on our list of item instances
  (length (items player))) )

;; Insert an item into the end of the  playlist
;; Args: pathname File or directory
;; 	 clear   True  - Insert a single item in the playlist, i.e. clear it before.
;;		 False - Append to the end of the playlistIf true.
;; Value: New item instance on success.
(defgeneric vendor-add (player pane &key pathname clear &allow-other-keys))

(defgeneric vendor-remove (player pane &key index item)
 ;;; Args: index,item Identify item to remove.
 (:method (player pane &key index item)
  (declare (ignore pane index item))
  (%not-implemented player 'vendor-remove)) )

(defgeneric vendor-clear (player pane)
 (:method (player pane)
  (declare (ignore player pane)))

 (:method :before (player pane)
  (when (item player)				; there is a current item
    (vendor-stop player pane :silent t)))

 (:method :after (player pane)
  (mapc #'release-item (items player))		; dangerous?
  (setf (items player) ()
        (index player) -1
        (state player) nil
        (status player) nil)) )

(defgeneric vendor-next (player pane)
 ;;; Args: force If true, play again in the signgleton playlist
 (:method (player pane)
  (%not-implemented player 'vendor-next)))

(defgeneric vendor-previous (player pane)
 (:method (player pane)
  (%not-implemented player 'vendor-previous)))

(defgeneric vendor-item-descripton (player pane &key item)
 (:method (player pane &key (item (item player)))
  (when item
    (format nil "Title ~a%Type ~s"
            (item-title item)
            (when-let (url (item-url item))
              (if (pathnamep url) (pathname-type url) nil))))) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  PROPERTIES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Methods on vendor-get and vendor-xx should extract a property value
;;; - either from the plist of a pane every time,
;;; - or from a COM-object.
;;;   The latter COM-properties should be initialized in component-on-insert like this:
;;; 	(setf (vendor-xx player pane) (getf :xx (CAPI:CAPI-OBJECT-PLIST pane) *xx*)
;;;
;;; :AUTOLOOP
;;;	On end, restrart playing the playlist from the beginning
;;; :AUTOSTART
;;;	On appending the first item to the playlist, start playing it,
;;;	i.e. affects only adding the very first item.
;;;	On ivoking next/previous, autostart does not matter and autoplay
;;;	depends on the vendor:
;;;	- VLS aways starts,
;;;	- WMP depending on the current status.
;;; :RATE
;;;	Playback speed (and direction?)
;;;	>0  Forward:
;;;	1	normal speed,
;;;	>1	fast, e.g. 2.0 twice as fast,
;;;	<1	slow, e.g. 0.5 for half speed.
;;;	<0  backward (TODO?)
;;; :SINGLETON
;;; 	Acctually the property of the player pane: if true only one item is allowed.

(defgeneric vendor-get (player pane prop &optional default)
 ;;; Args: prop ::=  :autoloop | :autostart | :rate
 (:method (player pane prop &optional default)
  (declare (ignore player))
  (getf (CAPI:CAPI-OBJECT-PLIST pane) prop default))

 (:method (player pane (prop (eql :rate)) &optional (default 1.0))
  default) )

(defgeneric vendor-put (player pane prop value)
 (:method (player pane prop value)
  (declare (ignore playr))
  (setf (capi:capi-object-property pane prop) value)) )

(defsetf vendor-get (player pane prop &optional default) (value)
  (declare (ignore default))
  `(vendor-put ,player ,pane ,prop ,value))

(defgeneric vendor-ui-mode (player pane)
 (:method (player pane)
  :none) )

(defgeneric (setf vendor-ui-mode) (value player pane &key  insert)
 ;;; Args: insert True if called by vendor-component-on-insert
  ;; 		  False if called afterwards
 (:method (value player pane &key insert)
  (declare (ignore pane insert))
  (%not-implemented player '(setf vendor-ui-mode))) )

(defun timestring (sec)
 ;;; Helper: Format duration or current time in "hH:MM:SS" format.
  ;;	     If the time is less than an hour long, the "hH:" portion is not included.
  ;; Args: sec Integer number of seconds
  (multiple-value-bind (seconds minutes hours) (decode-universal-time sec 0)
    (format nil "~:[~*~;~d:~]~2,'0d:~2,'0d"
            (< 0 hours) hours minutes seconds)))

(defgeneric vendor-duration (player &optional pane i-media)
 (:method (player &optional pane i-media)
  nil) )

(defgeneric (setf vendor-duration) (sec player &optional pane i-media)
 ;;; Would be useful for photo
 (:method (sec player &optional pane i-media)
  (%not-implemented player '(setf vendor-duration))) )

(defgeneric vendor-duration-string (player &optional pane i-media)
 (:method (player &optional pane (i-media nil specified-p))
  (let ((sec (if specified-p
                 (vendor-duration player pane i-media)
                 (vendor-duration player pane))))		; let it use default media
    (if sec (timestring sec) ""))))

(defgeneric vendor-time (player &optional pane)
 (:method (player &optional pane)
  nil)

 (:method :around (player &optional pane)
  ;; Memoize the result within the current item instance
  (let ((sec (call-next-method)))
    (when sec
      (when-let (item (item player))
        (setf (item-time item) sec)))
    sec)) )

(defgeneric (setf vendor-time) (sec player &optional pane)
 (:method (value player &optional pane)
  (%not-implemented player '(setf vendor-tiem))) )

(defgeneric vendor-timestring (player &optional pane)
 (:method (player &optional pane)
  (let ((sec (vendor-time player pane)))
    (if sec (timestring sec) ""))) )

(defgeneric vendor-position (player &optional pane)
 ;; Position is usaally a double float in the [0.0-1.0] range
 (:method (player &optional pane)
  nil) )						; %not-implemented?

(defgeneric (setf vendor-position) (value player &optional pane)
 (:method (value player &optional pane)
  (%not-implemented player '(setf vendor-position))) )

(defgeneric vendor-state (player &optional pane index)
 ;;; Current play/open state as a keyword or or NIL
  ;; Args: index Vendor-specific integer status code aquired from the COM-object.
 (:method (player &optional pane index)
  nil) )

(defgeneric vendor-status (player &optional pane)
 ;;; Current state as a string (localized) if available or NIL
 (:method (player &optional pane)
  nil))

(defgeneric vendor-version (player &optional pane)
 (:method (player &optional pane)
  nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  METHODS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The below four are obligatory as well as vendor-add

;; Args: index, item  Force playing from the specified item.
(defgeneric vendor-play (player pane &key index item))

;; Args: on If specified, change accordingly; if not specified, toggle.
(defgeneric vendor-pause (player pane &key on))

(defgeneric vendor-rewind (player pane))

;; Release some resuources but does not change the current item.
;; If after stop play is called again, it starts with opening, buffering etc.
(defgeneric vendor-stop (player pane  &key silent))

(defgeneric vendor-fast-backward (player pane &key rate)
 (:method (player pane &key rate)
  (declare (ignore pane rate))
  (%not-implemented player 'vendor-fast-backward)))

(defgeneric vendor-fast-forward (player pane &key rate)
 (:method (player pane &key rate)
  (declare (ignore pane rate))
  (%not-implemented player 'vendor-fast-forward)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  EVENTS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro define-event (vendor-id medhod-name fn &rest args)
 ;;; Args: method-name	String according to the docs
  ;;	   fn		Function of signature: (pane vector [&ptional|&rest] arg*),
  ;;			where vector is a simple vector.
  ;;	   args		When specified, passed to fn aftr the vector.
  ;; Neither method-name nor fn nor args are evaluated.
  ;(lw:with-unique-names (vendor)				; for top-level usage only
  `(when-let (vendor (find-vendor ,vendor-id))
     (setf (gethash ,medhod-name (vendor-events vendor))
           ',(if args `(,fn . ,args) `,fn))))

(defmacro get-event-handler (vendor-id method-name)
  `(gethash ,method-name (vendor-events (find-vendor ,vendor-id))))

(defun sink-callback (pane method-name kind vector)
 ;;; Default callback function responding to events of COM-interface name :default
  ;; Args: kind ::= :method
  ;;		  | :get | :put - seem never passed by VLC or WMP.
  ;;       vector  Simple vector, may be empty
  ;;		   CAUTION: This vector seems to be reused by LW, so always store a copy!
  #-debug (declare (ignore kind))				;(when (eq kind :method)
  (let* ((vendor-id (slot-value (player pane) 'vendor-id))
         (handler (get-event-handler vendor-id method-name)))
    (cond (handler
           #+ys-product
           ;(unless (or (equal method-name "StatusChange")		; WMP
           ;            (equal method-name "MediaPlayerTimeChanged"))	; VLC
           (yl:debug-format :media "~a handling event ~s ~s"
                            vendor-id method-name vector)
           (if (consp handler)
               (apply (first handler) pane vector (rest handler))	; pass extra args
               (funcall handler pane vector)))				; no extra args
          #+debug
          ((or (equal method-name "MouseMove")				; do not log these
               ;(equal method-name "MouseUp") ;(equal method-name "MouseDown")
               ;(equal method-name "Click")
               (equal method-name "KeyDown") (equal method-name "KeyUp")
               (equal method-name "Buffering")				; WMP
               (equal method-name "MediaPlayerBuffering")		; VLC
               (equal method-name "PlaylistChange")			; WMP
               (equal method-name "MediaPlayerPositionChanged")		; VLC
          ))
          #+ys-product
          (t (yl:debug-format :media "~a unhandled event ~s (~s) ~s"
                              vendor-id method-name kind vector)) )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  TOP-LEVEL API  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+unused
(defgeneric state (player &optional pane)
 ;;; Vendor-dependend methods can extract actual value from com-object and put to cache.
  ;; Values: 1) NIL or keyword,
  ;;	     2) NIL or localized string if available.
 (:method (player &optional pane)
  (values (vendor-state player pane) (vendor-status player pane))) )

;;; EVENT CALLBACKS

(defgeneric on-open (interface pane &optional item)	; start playing
 (:method (interface pane &optional item)) )

(defgeneric on-play (interface pane)
 (:method (interface pane)))				; start or resume

(defgeneric on-pause (interface pane)
 (:method (interface pane)))

(defgeneric on-end (interface pane)			; end reached
 (:method (interface pane)) )

(defgeneric on-stop (interface pane)
 (:method (interface pane)) )


(defgeneric on-state-change (face pane &key &allow-other-keys)
 ;;; Args: status   keyword
  ;;		    T		Acquire by venodr-state
  ;;	   status   string or NIL
  ;;		    T		Acquire by venodr-status
  ;;	   time     integer	Current second from beginning
  ;;	   	    T		Acquire by venodr-time
  ;;	   duration integer	Length is seconds.
  ;;	   position float	Double float in [0..1] range (VLC only).
  ;; NB: If any of the argumets is NIL, its value should be extracted from 
  ;;	 the player and current item instances, not from the COM-interface.
 (:method (interface pane &key)))

;;; ERRORS

(defgeneric on-error (interface pane error &rest args)
 (:method (interface pane error &rest args)
  (apply #'capi:display-message error args))

 #+(and ys-product debug)
 (:method :before (interface pane error &rest args)
  (apply #'yl:debug-format :media error args)) )

(defmethod capi:report-active-component-failure ((pane player-pane) component-name
                                                 error function hresult)
  (let ((s (format nil "Player ActiveX failure:~% component ~s~% function ~s => ~s~%~a."
                   component-name function hresult error)))
    #+ys-product (yl:debug-format :media s)
    (capi:display-message s)))


(defun print-event-hanlers (vendor &optional (stream t))
  (format stream "~%EVENT HANLDERS FOR ~s" vendor)
  (loop for key being each hash-key in (vendor-events vendor)
        using (hash-value value)
        do (format stream "~% ~s -> ~s" key value)))
  
#||
;;; Background not working in any implementation
(defun integer-rgb-color (integer)
 ;;; Args: integer  A la CSS #CCBBFF but specified as a Lisp number like #xCCBBFF
  (color:make-rgb (/ (ldb (byte 8 16) integer) 255.0)
                  (/ (ldb (byte 8 8) integer) 255.0)
                  (/ (ldb (byte 8 0) integer) 255.0)))

(defun rgb-integer (red green blue)
  (dpb (round (* red #255=255.0)) (byte 8 16)
       (dpb (round (* green #255#)) (byte 8 8)
            (dpb (round (* blue #255#)) (byte 8 0) 0))))

(defgeneric vendor-background (player &optional pane)
 ;;; Keyworkd or RGB-color
 (:method (player &optional pane)
  :black) )

(defgeneric (setf vendor-background) (value player &optional pane)
 (:method (value player &optional pane)
  (%not-implemented player '(setf vendor-background))) )
||#
